/*  Erneute Terminierung des gegeben Arbeitsgangs mit erlaubter Fragmentierung. Anschließend wird die Dauer eventuelle
    aufgetrtener Überlappungen mit anderen AGen blockiert.

    Für die Terminierung wird die ursprüngliche Hauptressource und der ursprüngliche Startzeitpunkt verwendet.

    Fragmentierung bedeuted, dass Task-Einträge anderer AGe die Abarbeitung des aktuellen AG unterbrechen dürfen. Ab der ersten
    Fragmentierung werden nur noch Blockierungen ('task.blocktime') für den gegebenen AG gebildet.
*/
SELECT tsystem.function__drop_by_regex( 'resource_timeline__ab2__retermination_fragmented', 'scheduling', _commit => true );
CREATE OR REPLACE FUNCTION scheduling.resource_timeline__ab2__retermination_fragmented(
      _ab2                    ab2,                                    -- Zu terminierender AG
      _timeframe_start        timestamp DEFAULT null,                 -- Minimum des Zeitbereichs, in dem die ABK einterminiert werden soll. Passt die ABK nicht in den Zeitbereich, schlägt Terminierung mit Exception fehl. Überschreibt den ursprünglichen Beginn als Strattermin für Reterminierung.
      _timeframe_end          timestamp DEFAULT null,                 -- Maximum des Zeitbereichs, in dem die ABK einterminiert werden soll. Passt die ABK nicht in den Zeitbereich, schlägt Terminierung mit Exception fehl.
      _blocktime_date_start   timestamp DEFAULT null,                 -- Startzeitpunkt, ab dem Blockierungen eingetragen werden sollen.
      _loglevel               int DEFAULT TSystem.Log_Get_LogLevel( _user => 'yes' )
  ) RETURNS bool AS $$
  DECLARE
      _prefix   varchar := format( 'resource_timeline__ab2__retermination__fragmented AG %L -', _ab2.a2_id );

      _resource_id_main           int;
      _resource_id_fixed          int;
      _current_time               timestamp := coalesce( nullif( TSystem.Settings__Get( 'CI.AssumedNow' ), '' )::timestamp, localtimestamp( 0 ) );   -- Aktueller Zeitpunkt
      _is_dlz                     bool := false;                      -- Aktueller AG wurde mit 'force dlz' terminiert
      _is_parallel                bool := false;                      -- Anderer AG ist parallel und aktueller AG ist daruch nicht zum vorgegebenen Stratzeitpunkt terminierbar.
      _is_future_and_no_parallel  bool := true;                       -- AG liegt in Zukunft und hat keinen mit "force dlz" terminierten AG zeitlich parallel, dann kann Reterminiert werden.
      _work_required              numeric;                            -- Noch notwendige bzw. offene Laufzeit bzw. Arbeitszeit des AG.
      _work_scheduled_from_now    numeric;                            -- Eingeplante Laufzeit des AG ab aktuellem Zeitpunkt.
      _work_required_missing      numeric := 0;                       -- Fehlender Rest der Laufzeit des AG. Dieser Rest ist noch nicht eingeplant, da nur bis zum ersten Auftreten von Fragmentierung Terminiert werden sollte.
      _overlap_duration           numeric := 0;                       -- Dauer der Überlappungen
      _blocktime_duration         numeric;                            -- Gesamtdauer der zu bildenden Blockierungen. Ist die Summe aus fehlenden Stempelungen und Überlappungen.
      _min_blocktime_duartion CONSTANT numeric := 900;                -- Minimale Dauer bevor Blockierungen gebildet werden. Aktuell 15 Minuten.

  BEGIN
      IF _loglevel >= 4 THEN
          RAISE NOTICE '%', format(
                                      'call: scheduling.resource_timeline__ab2__retermination_fragmented( ab2 => %L , _timeframe_start => %L, _timeframe_end => %L, _timeframe_start => %L, _blocktime_date_start => %L, _loglevel => %L )',
                                                                                                          _ab2.a2_id, _timeframe_start      , _timeframe_end      , _timeframe_start      , _blocktime_date_start      , _loglevel
                                  )
          ;
      END IF;

      -- Debug.
      IF _loglevel >= 5 THEN
          RAISE NOTICE '% now:%;', _prefix, _current_time;
      END IF;

      -- Prüfen, ob AG exisiert.
      IF NOT EXISTS ( SELECT true FROM ab2 WHERE a2_id = _ab2.a2_id ) THEN
          RAISE NOTICE '% AG not found!', _prefix;
          RETURN false;
      END IF;

      -- Hauptressource bestimmen.
      _resource_id_main := a2w_resource_id_main_terminated
                      FROM ab2_wkstplan
                     WHERE a2w_a2_id = _ab2.a2_id
                       AND NOT a2w_marked = -1;

      -- Prüfen, ob AG aktuell einterminiert ist.
      IF NOT EXISTS (
          SELECT true
            FROM scheduling.resource_timeline
            WHERE ti_a2_id = _ab2.a2_id
              AND ti_resource_id = _resource_id_main
              AND ti_type IN ( 'task', 'task.buffer' )
      ) THEN
          RAISE NOTICE '% AG currently not terminated!', _prefix;
          RETURN false;
      END IF;

      -- Startzeitpunkt bestimmen. Entweder vorgegebener oder ursrünglicher Start.
      _timeframe_start := coalesce (
                                        _timeframe_start  -- Wenn Parameter "_timeframe_start" vorgegeben, diesen Startzeitpunkt verwenden, ...
                                      , _ab2.a2_at        -- ... ansonsten den ursrünglichen Startzeitpunkt verwenden.
                                   )
      ;
      -- Endzeitpunkt bestimmen. Entwerder vorgegeben oder 1 Jahr nach dem Start.
      _timeframe_end := coalesce( _timeframe_end, _timeframe_start + interval '1 year' );
      -- Prüfung Start vor Ende.
      IF ( _timeframe_end < _timeframe_start ) THEN
          RAISE EXCEPTION USING
              ERRCODE = 'TFAIL',
              MESSAGE = '_timeframe_end < _timeframe_start : "' || _timeframe_end::varchar || '<' || _timeframe_start::varchar || '"',
              HINT = 'Termination not possible within given timeframe';
      END IF;

      -- Startzeitpunkt festlegen, ab dem Blockierungen eingetragen werden sollen.
      _blocktime_date_start := coalesce( _blocktime_date_start, greatest( _ab2.a2_et, _current_time ) ); -- Vorgabe, falls vorhanden oder Ende ds AG. Aber mindestens der jetzige Zeitpunkt.

      -- Prüfung auf ursprünliche DLZ-Terminierung.
      IF  EXISTS (
              SELECT true
                FROM scheduling.resource_timeline
              WHERE ti_a2_id = _ab2.a2_id
                AND ti_resource_id = _resource_id_main
                AND ti_type IN ( 'task', 'task.buffer' )
                AND ti_stat = 'dlz'
          )
      THEN
          _is_dlz := true; -- AG war mit force dlz terminiert
      END IF;

      -- Der AG liegt teilweise in Vergangenheit, dann ...
      IF _ab2.a2_at < _current_time THEN

          -- ... liegt er natürlich NICHT vollständig in der Zukunft!
          _is_future_and_no_parallel := false;

      -- Der AG liegt vollständig in der Zukunft, dann ...
      ELSE

          -- ... prüfen, ob er einen mit "force dlz" terminierten AG zeitlich parallel hat.
          IF      _is_dlz IS false
              AND EXISTS (
                      SELECT true
                        FROM ab2 AS o
                      WHERE a2_id  <> _ab2.a2_id  -- anderer AG
                        AND EXISTS(               -- mit zugehörigen Timeline-Eintrag welcher vom Status 'dlz' ist
                                      SELECT true
                                        FROM scheduling.resource_timeline AS i
                                        WHERE     i.ti_a2_id = o.a2_id
                                              AND i.ti_resource_id = _resource_id_main
                                              AND i.ti_type IN ( 'task', 'task.buffer' )
                                              AND i.ti_stat = 'dlz'
                                  )
                        AND o.a2_at < _ab2.a2_et  -- zeitliche Überlappung
                        AND o.a2_et > _ab2.a2_at  -- zeitliche Überlappung
                  )
          THEN
              _is_parallel                := true;
              _is_future_and_no_parallel  := false;
          END IF;

      END IF;

      -- AG liegt in Zukunft und hat keinen mit "force dlz" terminierten AG zeitlich parallel, dann ...
      IF _is_future_and_no_parallel THEN
          -- ... diesen Neuterminieren.

          -- Ursprünglich fixierte Ressource merken.
          _resource_id_fixed := a2w_resource_id_main_fix
                        FROM ab2_wkstplan
                      WHERE a2w_a2_id = _ab2.a2_id
                        AND a2w_marked >= 0;

          -- Aktuelle Haupressource fixieren.
          PERFORM scheduling.ab2_wkstplan__resource_id_main_fix__move__set(
                      _a2_id                    => _ab2.a2_id,
                      _to_resource_id           => _resource_id_main,
                      _all_ag_withthis_resource => false,
                      _loglevel                 => _loglevel
                  )
          ;

          -- Terminierung des AG entfernen.
          DELETE
            FROM scheduling.resource_timeline
          WHERE ti_a2_id = _ab2.a2_id
          ;

          -- ... AG mit erlaubter Fragmentierung erneut terminieren.
          PERFORM scheduling.resource_timeline__abk_ab2__termination(
                      _abk_ix                         => _ab2.a2_ab_ix
                    , _timeframe_start                => _timeframe_start
                    , _timeframe_end                  => _timeframe_end
                    , _write_to_disk                  => true
                    , _direction                      => 'forward'
                    , _checkBlockedTimes              => NOT _is_dlz
                    , _termination_range_ab2_id_start => _ab2.a2_id
                    , _termination_range_ab2_id_end   => _ab2.a2_id
                    , _allow_overlap                  => true
                    , _resource_id_main_fix__set      => false
                    , _allow_fragmentation            => 'yes'
                    , _loglevel                       => _loglevel
                  )
          ;

          -- Ursprünglich fixierte Ressource wieder herstellen.
          PERFORM scheduling.ab2_wkstplan__resource_id_main_fix__move__set(
                      _a2_id                    => _ab2.a2_id,
                      _to_resource_id           => _resource_id_fixed,
                      _all_ag_withthis_resource => false
                  )
          ;

          -- Dauer der Überlappungen auf der Hauptressource mit anderen AGen ermitteln.
          IF _is_dlz IS true THEN
              _overlap_duration :=  scheduling.resource_timeline__overlap_duration__by__a2_id(
                                        _a2_id                => _ab2.a2_id
                                      , _filter_for_other_ab2 => 'dlz_a2_et'
                                    );
          END IF;

      -- AG liegt mindestens teilweise in der Vergangenheit oder hat einen mit "force dlz" terminierten AG zeitlich parallel, dann ...
      ELSE
          -- ... den AG so belassen, NICHT neu terminieren und Blockierungen für Verzug und Vergrößerung der Vorgabearbeitszeit bilden.

          -- Debug.
          IF _loglevel >= 5 THEN
              RAISE NOTICE '% Skipping retermination (AG parallel:%, a2_at:%, now:%), just create Blocktimes.', _prefix, _is_parallel, _ab2.a2_at, _current_time;
          END IF;

          -- vorhergende Blockierunen des AG löschen.
          DELETE
          FROM scheduling.resource_timeline
          WHERE     ti_a2_id = _ab2.a2_id
                AND ti_type = 'task.blocktime';

          -- ... aber die fehlende Laufzeit Blockieren.
          -- Fehlende Laufzeit errechnen.
          _work_required := scheduling.ab2__required_worktime__get( _ab2_id => _ab2.a2_id );
          _work_scheduled_from_now := scheduling.resource_timeline__ab2__get_scheduled_time(
                                          _ab2_id                         => _ab2.a2_id
                                        , _ti_types                       => ARRAY[ 'task', 'task.buffer', 'task.blocktime' ]::scheduling.resource_timeline_blocktype[]
                                        , _timeframe_start                => _current_time
                                        , _resource_id                    => _resource_id_main
                                      )
          ;
          _work_required_missing := _work_required - _work_scheduled_from_now;
          -- Debug
          IF _loglevel >= 5 THEN
              RAISE NOTICE '% _work_required_missing(%) = _work_required(%) - _work_scheduled_from_now(%)', _prefix, round( _work_required_missing / 3600, 2 ), round( _work_required / 3600, 2 ), round( _work_scheduled_from_now / 3600, 2 );
          END IF;

          -- Dauer der Überlappungen auf der Hauptressource mit anderen AGen ermitteln.
          IF _is_dlz IS true THEN
              _overlap_duration :=  scheduling.resource_timeline__overlap_duration__by__a2_id(
                                        _a2_id                => _ab2.a2_id
                                      , _filter_for_other_ab2 => 'dlz_a2_et'
                                    );
          END IF;

      END IF;

      -- Entsprechende Zeiten ab frühestem aktuellen Zeitpunkt blockieren.
      _blocktime_duration := _work_required_missing + _overlap_duration;
      -- Debug
      IF _loglevel >= 5 THEN
          RAISE NOTICE '% _blocktime_duration(%) = _work_required_missing(%) - _overlap_duration(%)', _prefix, round( _blocktime_duration / 3600, 2 ), round( _work_required_missing / 3600, 2 ), round( _overlap_duration / 3600, 2 );
      END IF;
      --
      IF _blocktime_duration > _min_blocktime_duartion THEN
          PERFORM scheduling.resource_timeline__ab2__block_duration(
                      _ab2_id               => _ab2.a2_id
                    , _blocktime_duration   => _blocktime_duration
                    , _blocktime_date_start => _blocktime_date_start
                    , _loglevel             => _loglevel
                  )
          ;
      ELSE
          IF _loglevel >= 5 THEN
              RAISE NOTICE '% Skipping creation of Blocktimes: _blocktime_duration(%) < _min_blocktime_duartion(%)', _prefix, round( _blocktime_duration / 3600, 2 ), round( _min_blocktime_duartion / 3600, 2 );
          END IF;
      END IF;

      RETURN true;

  END $$ LANGUAGE plpgsql;
--